﻿var pewModule = require('./pew_module');
var pew = new pewModule.pewModule();
var Mew = require('./mewtocol');
var Mew7 = require('./mewtocol7');
var Modbus = require('./modbus');
const tls = require("tls");
const fs = require("fs");
const path = require('path');
var net = require('net');

var client, server;
var serverSocket = [];
var serverSocketIndex = 0;
var protocol;
const moduleName = "TLS Server";
const accessModes = {
    localhostOnly: 0,
    singleIP: 1,
    IPRange: 2
}
const authMethods = {
    publicKey: 0,
    clientCert: 1
}

const STATUS = {
    noErr: 0,
    tlsClientConnected: 1,
    tlsServerDisconnected: 2,
    serverConnectErr: -1,
    portOccupiedErr: -2,
    serverConnectTimeout: -3,
    missingCertErr: -4,
    remoteIPDenied: -5,
    unexpectedErr: -99
}
var usePort, useProtocol, serviceID;
var tlsConfig, portConfig, interfaceConfig;
var connected = false;
var thread, listeningServer;
//Catch exceptions
process.on('uncaughtException', function (err) {
    pew.sysLogMessage(moduleName, "Uncaught exception: " + err.stack);
})

initialize();

plcPolling();

function connectToServer() {
    return new Promise((res, rej) => {
        try {
            let interval; 
            client = new net.Socket();

            interval = setTimeout(() => {
                setPLCStatus([STATUS.noErr, STATUS.serverConnectTimeout]);
                connected = false;
                pew.sysLogMessage(moduleName, "Connect to server timeout");
            }, pew.Constants.Timeout_Status_request)

            client.connect(tlsConfig.data.nontls_destination_port, tlsConfig.data.nontls_server_host, function () {
                connected = true;
                clearTimeout(interval);
                res();
            });

            client.on('data', function (data) {
                if (serverSocket.length > 0) {
                    serverSocket[serverSocketIndex].write(data);
                }
            });

            client.on("error", err => {
                setPLCStatus([STATUS.noErr, STATUS.serverConnectErr]);
                pew.sysLogMessage(moduleName, `Connection error: ${err}`);
                connected = false;
                disconnectFromServer();
                clearTimeout(interval);
                rej();
            })

            client.on("timeout", () => {
                setPLCStatus([STATUS.noErr, STATUS.serverConnectTimeout]);
                connected = false;
                pew.sysLogMessage(moduleName, "Connect to server timeout");
            })

            client.on('close', function () {
                clearTimeout(interval);
                connected = false;
            });
        }
        catch (ex) {
            setPLCStatus([STATUS.noErr, STATUS.unexpectedErr]);
            connected = false;
            pew.sysLogMessage(moduleName, ex.message);
            clearTimeout(interval);
            rej();
        }
    })
}

function setPLCStatus(statusValues) {
    let setStatus = {
        area: "DT",
        type: "INT",
        count: 2,
        start: tlsConfig.data.plc_status_dt,
        value: statusValues
    }
    protocol.StartAddr = setStatus.start;
    //Service 5 is Ethernet. In case of ethernet interface, don't use the station number,
    //instead use EE
    if (thread[0].service !== 5) {
        protocol.Station = interfaceConfig.data.address_plc;
    }   

    pew.writeMultipleRegisters(protocol, usePort, setStatus).catch(err => {
        pew.sysLogMessage(moduleName, err.err_msg);
    });
}

function disconnectFromServer() {
    try {
        client.end();
        client.destroy();
    }
    catch (ex) {
        pew.sysLogMessage(moduleName, ex.message);
    }
}

function initialize() {
    //Read Configurations
    interfaceConfig = pew.getConfFileSync(pew.Constants.requests.READ_INTERFACE);
    portConfig = pew.getConfFileSync(pew.Constants.requests.READ_PORTS);
    tlsConfig = pew.getConfFileSync(pew.Constants.requests.READ_TLS);

    //If any configuration could not be loaded -> log the message and stop execution
    if (tlsConfig.err) {
        pew.sysLogMessage(moduleName, tlsConfig.err_msg);
        return;
    }

    //Check configurations
    serviceID = interfaceConfig.data.interface_plc;
    let iface = interfaceConfig.data.interface.filter(val => {
        return val.service == serviceID;
    })

    useProtocol = iface[0].protocol;
    //Set the protocol module
    switch (useProtocol) {
        case pew.Constants.PROTOCOLS.mewtocol:
            protocol = new Mew.Mewtocol();
            break;

        case pew.Constants.PROTOCOLS.mewtocol7:
            protocol = new Mew7.Mewtocol7();
            break;

        case pew.Constants.PROTOCOLS.modbus:
            protocol = new Modbus.Modbus();
            break;

        default:
            protocol = new Mew.Mewtocol();
    }

    thread = portConfig.data.thread.filter(val => {
        return val.service === serviceID;
    });

    if (thread.length <= 0) {
        pew.sysLogMessage(moduleName, pew.Constants.TCP_PORT_DISABLED);
        return false;
    }
    usePort = thread[0].port;

    if (pew.isPortOccupied(tlsConfig.data.tls_server_port)) {
        setPLCStatus([STATUS.noErr, STATUS.portOccupiedErr]);
        pew.sysLogMessage(moduleName, `${pew.Constants.TCP_PORT_OCCUPIED} -- ${tlsConfig.data.tls_server_port}`);
        return false;
    }

    if (tlsConfig.data.tlsserverca === pew.Constants.EMPTY_STRING || tlsConfig.data.tlsservercert === pew.Constants.EMPTY_STRING || tlsConfig.data.tlsserverkey === pew.Constants.EMPTY_STRING) {
        setPLCStatus([STATUS.noErr, STATUS.missingCertErr]);
        pew.sysLogMessage(moduleName, pew.Constants.CERT_FILE_MISSING);
        return false;
    }

    //Set minimum TLS Version
    tls.DEFAULT_MIN_VERSION = tlsConfig.data.tls_min_version_server;

    listeningServer = tlsConfig.data.tls_accessability == accessModes.localhostOnly ? pew.Constants.LOCAL_HOST : pew.Constants.ALL_IP;
    /***************************************************************************************************************************/

    /********************************************************************************/
    //Server - encrypted
    /********************************************************************************/
    const options = {
        isServer: true,
        key: fs.readFileSync(path.join(pew.Constants.configSubfolders.tls, tlsConfig.data.tlsserverkey)),
        cert: fs.readFileSync(path.join(pew.Constants.configSubfolders.tls, tlsConfig.data.tlsservercert)),
        ca: fs.readFileSync(path.join(pew.Constants.configSubfolders.tls, tlsConfig.data.tlsserverca)),
        rejectUnauthorized: tlsConfig.data.tlsserver_certauth === authMethods.clientCert ? true : false,
        requestCert: tlsConfig.data.tlsserver_certauth === authMethods.clientCert ? true : false
    };

    server = tls
        .createServer(options, (socket) => {            
            let closeConnection = true;
            //Check if IP accessibility is limited
            switch (tlsConfig.data.tls_accessability) {
                //IN case of localhost, devices are only able to connect from localhost
                case accessModes.localhostOnly:
                    closeConnection = false;
                    break;
                case accessModes.singleIP:
                    if (socket.remoteAddress === tlsConfig.data.tls_access_ip1)
                        closeConnection = false;
                    break;

                case accessModes.IPRange:
                    let remoteIP = pew.ip2int(socket.remoteAddress);
                    let ip1 = pew.ip2int(tlsConfig.data.tls_access_ip1);
                    let ip2 = pew.ip2int(tlsConfig.data.tls_access_ip2);
                    if (remoteIP >= ip1 && remoteIP <= ip2)
                        closeConnection = false;
                    break;
            }

            if (closeConnection) {
                socket.end();
                setPLCStatus([STATUS.noErr, STATUS.remoteIPDenied]);
                pew.sysLogMessage(moduleName, `Remote IP (${socket.remoteAddress}) not allowed`);
            }            

            //Set idle timeout, if there is an inactivity for this time period
            //the connection will be ended
            socket.setTimeout(pew.Constants.IDLE_TIMEOUT, () => {
                socket.end();
            })
            serverSocket.push(socket);
            socket.on("data", data => {
                //Check which connection send the data and set the index
                for (let i = 0; i < serverSocket.length; i++) {
                    if (
                        serverSocket[i].remoteAddress === socket.remoteAddress &&
                        serverSocket[i].remotePort === socket.remotePort
                    ) {
                        serverSocketIndex = i;
                        break;
                    }
                }

                //Check if client is connected                
                if (connected) {
                    client.write(data);
                }
                else {
                    connectToServer().then(() => {
                        client.write(data);
                    }).catch(err => {
                        pew.sysLogMessage(moduleName, err);
                    });
                }
            })

            //If a connection get closed, remove it also from the socket list
            socket.on("close", () => {
                serverSocket = serverSocket.filter((val) => {
                    return !(
                        socket.remoteAddress === val.remoteAddress &&
                        socket.remotePort === val.remotePort
                    );
                });

                if (serverSocket.length === 0) {
                    setPLCStatus([STATUS.noErr, STATUS.noErr]);
                }
            })
        })
        .on("connection", () => {
            setPLCStatus([STATUS.noErr, STATUS.tlsClientConnected]);
        })
        .listen(tlsConfig.data.tls_server_port, listeningServer, () => {
            setPLCStatus([STATUS.noErr, STATUS.noErr]);
            pew.sysLogMessage(moduleName, "Listening on port " + tlsConfig.data.tls_server_port + "\n");
        });

    /**************************************************s******************************/
    //Client
    /********************************************************************************/
    //Connect to NON TLS Server
    connectToServer().catch(err => {
        //Just in case the server is not reachable...
        pew.sysLogMessage(moduleName, err);
    });
}

function plcPolling() {
    let readItem = {
        area: "DT",
        type: "INT",
        count: 1,
        start: tlsConfig.data.plc_status_dt
    }

    const control = {
        connect: 1,
        disconnect: 2
    }

    protocol.StartAddr = readItem.start;
    pew.readMultipleRegisters(protocol, usePort, readItem).then(data => {
        if (!data.err) {
            switch (data.int[0]) {
                case control.connect:
                    if (server && !server._handle) {
                        server.listen(tlsConfig.data.tls_server_port, listeningServer);                        
                    }
                    setPLCStatus([STATUS.noErr, STATUS.noErr]);
                    break;

                case control.disconnect:
                    //Disconnect the server and also close all connections. 
                    //Attention: even if server is closed, established connections wont be disconnected! Manually disconnect them
                    if (server) {
                        //manually disconnect all connected clients
                        for (let con of serverSocket) {
                            con.destroy();
                        }
                        serverSocket = [];
                        server.close();
                    }
                    setPLCStatus([STATUS.noErr, STATUS.tlsServerDisconnected]);
                    break;
            }
        }
        else {
            pew.sysLogMessage(moduleName, data.err_msg);
        }
    }).catch(err => {
        pew.sysLogMessage(err);
    }).finally(() => {
        setTimeout(plcPolling, tlsConfig.data.plc_poll_tlsserver);
    })
}